<?php
/**
 * Plugin Name: Schoolwiki
 * Description: Schoolwiki search widget voor Wordpress-websites.
 * Version:     1.0.0
 * Author:      Infowijs
 * Author URI:  https://infowijs.nl/
 * Donate link: https://schoolwiki.nl/
 * License:     GPLv2
 * Text Domain: schoolwiki
 * Domain Path: /languages
 *
 * @link    https://schoolwiki.nl/
 *
 * @package Schoolwiki
 * @version 1.0.0
 */

/**
 * Copyright (c) 2018 Infowijs (email : info@infowijs.nl)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, version 2 or, at
 * your discretion, any later version, as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * Autoloads files with classes when needed.
 *
 * @since  1.0.0
 * @param  string $class_name Name of the class being requested.
 */
function schoolwiki_autoload_classes( $class_name ) {

	// If our class doesn't have our prefix, don't load it.
	if ( 0 !== strpos( $class_name, 'S_' ) ) {
		return;
	}

	// Set up our filename.
	$filename = strtolower( str_replace( '_', '-', substr( $class_name, strlen( 'S_' ) ) ) );

	// Include our file.
	Schoolwiki::include_file( 'includes/class-' . $filename );
}
spl_autoload_register( 'schoolwiki_autoload_classes' );

/**
 * Main initiation class.
 *
 * @since  1.0.0
 */
final class Schoolwiki {

	/**
	 * Current version.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	const VERSION = '1.0.0';

	/**
	 * URL of plugin directory.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $url = '';

	/**
	 * Path of plugin directory.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $path = '';

	/**
	 * Plugin basename.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $basename = '';

	/**
	 * Detailed activation error messages.
	 *
	 * @var    array
	 * @since  1.0.0
	 */
	protected $activation_errors = array();

	/**
	 * Singleton instance of plugin.
	 *
	 * @var    Schoolwiki
	 * @since  1.0.0
	 */
	protected static $single_instance = null;

	/**
	 * Instance of S_Options
	 *
	 * @since1.0.0
	 * @var S_Options
	 */
	protected $options;

	/**
	 * Creates or returns an instance of this class.
	 *
	 * @since   1.0.0
	 * @return  Schoolwiki A single instance of this class.
	 */
	public static function get_instance() {
		if ( null === self::$single_instance ) {
			self::$single_instance = new self();
		}

		return self::$single_instance;
	}

	/**
	 * Sets up our plugin.
	 *
	 * @since  1.0.0
	 */
	protected function __construct() {
		$this->basename = plugin_basename( __FILE__ );
		$this->url      = plugin_dir_url( __FILE__ );
		$this->path     = plugin_dir_path( __FILE__ );
	}

	/**
	 * Add hooks and filters.
	 * Priority needs to be
	 * < 10 for CPT_Core,
	 * < 5 for Taxonomy_Core,
	 * and 0 for Widgets because widgets_init runs at init priority 1.
	 */
	public function hooks() {
		add_action( 'init', array( $this, 'init' ), 0 );
	}

	/**
	 * Activate the plugin.
	 */
	public function _activate() {

	}

	/**
	 * Deactivate the plugin.
	 * Uninstall routines should be in uninstall.php.
	 */
	public function _deactivate() {
		// Add deactivation cleanup functionality here.
	}

	/**
	 * Init hooks
	 */
	public function init() {
      $this->register_options();

      $domain = get_option('schoolwiki_domain');

      if(empty($domain) && @$_GET['page'] != 'schoolwiki'){
        add_action('admin_init', function(){
          add_action( 'all_admin_notices', array( $this, 'requirements_not_met_notice' ) );
        });
      }else{
        add_action('wp_footer', function() use ($domain) {
          $config = ['domain' => $domain, 'wordpress' => 1];
          $options = ['school','title','position','color'];
          foreach($options as $option){
            $val = get_option('schoolwiki_' . $option);
            if($val){
              $config[$option] = $val;
            }
          }
          echo '

<!-- Start Schoolwiki widget (wp-'.$this->version.') -->

<script src="https://'.$domain.'.schoolwiki.nl/widget.js"></script>
<script>schoolwiki.init('.json_encode($config).');</script>
<style>iframe.schoolwiki-widget{margin:0 !important;}</style>

<!-- End Schoolwiki widget -->


';
        });
      }
	}

  public function admin_page() {
    ?>
    <div style="position: fixed; width: 540px; right: 0; top: 30px; bottom: 0; background-color: #5aa67f;">
      <iframe style="border: 0; width: 100%; height: 100%;" scrolling="no" frameborder="0" id="schoolwiki-widget-preview"></iframe>
    </div>
    <div style="margin-right: 540px; padding: 48px;">
    <div class="wrap">
    <h1>Schoolwiki Widget</h1>

    <p>
      De Schoolwiki widget is een zoekknop die u op uw website kunt plaatsen om gebruikers in staat te
      stellen direct te zoeken binnen de Schoolwiki van uw school.
    </p>

    <p>Voer hieronder uw subdomein en optioneel een schoolnaam in om het widget te activeren. Terwijl u typt ziet u hiernaast een voorbeeld van het widget met uw instellingen.</p>
    <br />
    <a class="button" href="mailto:info@infowijs.nl?subject=Schoolwiki+widget">Contact opnemen met helpdesk</a>


    <br /><br /><br /><hr /><br />

    <form method="post" id="schoolwiki-settings-form" action="options.php">
        <?php settings_fields( 'schoolwiki' ); ?>
        <?php do_settings_sections( 'schoolwiki' ); ?>
        <table class="form-table">
            <tr valign="top">
            <th scope="row">Subdomein <span style="color:red;">*</span></th>
            <td><input type="text" placeholder="mijnkoepel" required name="schoolwiki_domain" value="<?php echo esc_attr( get_option('schoolwiki_domain') ); ?>" /></td>
            </tr>

            <tr valign="top">
            <th scope="row">Naam school</th>
            <td><input type="text" placeholder="infowijscollege" name="schoolwiki_school" value="<?php echo esc_attr( get_option('schoolwiki_school') ); ?>" /></td>
            </tr>

            <tr valign="top">
            <th scope="row">Kleurcode</th>
            <td><input type="text" name="schoolwiki_color" placeholder="#000000" value="<?php echo esc_attr( get_option('schoolwiki_color') ); ?>" /></td>
            </tr>

            <tr valign="top">
            <th scope="row">Extra titel</th>
            <td><input type="text" name="schoolwiki_title" value="<?php echo esc_attr( get_option('schoolwiki_title') ); ?>" /></td>
            </tr>
            <tr valign="top">
            <th scope="row">Positie</th>
            <td>
              <input type="text" name="schoolwiki_position" placeholder="'left' of 'right'" value="<?php echo esc_attr( get_option('schoolwiki_position') ?: 'right' ); ?>" />
            </td>
            </tr>
        </table>
        <?php submit_button('Widget-instellingen opslaan'); ?>
    </form>
    </div>

    <script>
      (function(){
        var $ = jQuery;
        $(function(){
          function schoolwiki_update_preview(){
            var config = {demo:1,wordpress:1};
            $('#schoolwiki-settings-form input').each(function(){
              var nm = $(this).attr('name');
              if(nm && nm.indexOf('schoolwiki_') === 0){
                config[nm.substring(11)] = $(this).val();
              }
            });

            if(config.domain && config.domain.length){
              $('#schoolwiki-widget-preview').attr('src', 'https://' + config.domain + '.schoolwiki.nl/widget/search?config=' + JSON.stringify(config));
            }
          }

          $('#schoolwiki-settings-form input').change(schoolwiki_update_preview);
          schoolwiki_update_preview();
        });
      })();
    </script>

  </div>
    <?php
  }

  public function register_options () {
    add_action('admin_menu', function() {
        //create new top-level menu
        add_menu_page('Schoolwiki instellingen', 'Schoolwiki', 'administrator', 'schoolwiki', [$this, 'admin_page'] , 'data:image/svg+xml;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiPz4KPHN2ZyB3aWR0aD0iMjhweCIgaGVpZ2h0PSIyOHB4IiB2aWV3Qm94PSIwIDAgMjggMjgiIHZlcnNpb249IjEuMSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIiB4bWxuczp4bGluaz0iaHR0cDovL3d3dy53My5vcmcvMTk5OS94bGluayI+CiAgICA8ZyBpZD0iQXJ0Ym9hcmQtQ29weS0yIiBzdHJva2U9Im5vbmUiIHN0cm9rZS13aWR0aD0iMSIgZmlsbD0ibm9uZSIgZmlsbC1ydWxlPSJldmVub2RkIj4KICAgICAgICA8ZyBpZD0iR3JvdXAtMiIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoNC4wMDAwMDAsIDQuMDAwMDAwKSIgZmlsbD0iI0VFRUVFRSI+CiAgICAgICAgICAgIDxnIGlkPSJHcm91cCI+CiAgICAgICAgICAgICAgICA8ZyBpZD0iR3JvdXAtQ29weSI+CiAgICAgICAgICAgICAgICAgICAgPHBhdGggZD0iTTIwLDguOTk5ODY2MzYgQzIwLDQuMDI5MzcyMDIgMTUuOTcwODM1NCwwIDExLDAgQzYuMDI5NDMxODYsMCAyLDQuMDI5MzcyMDIgMiw4Ljk5OTg2NjM2IEMyLDEzLjk3MDM2MDcgNi4wMjk0MzE4NiwxOCAxMSwxOCBDMTUuOTcwODM1NCwxOCAyMCwxMy45NzAzNjA3IDIwLDguOTk5ODY2MzYgTTE4LjEyODk1MzcsOC45OTk4NjYzNiBDMTguMTI4OTUzNywxMi45MzA4NzYzIDE0LjkzMDgwMSwxNi4xMjg5ODE1IDExLDE2LjEyODk4MTUgQzcuMDY5MTk5MDEsMTYuMTI4OTgxNSAzLjg3MTA0NjMsMTIuOTMwODc2MyAzLjg3MTA0NjMsOC45OTk4NjYzNiBDMy44NzEwNDYzLDUuMDY5MTIzNzQgNy4wNjkxOTkwMSwxLjg3MTAxODUyIDExLDEuODcxMDE4NTIgQzE0LjkzMDgwMSwxLjg3MTAxODUyIDE4LjEyODk1MzcsNS4wNjkxMjM3NCAxOC4xMjg5NTM3LDguOTk5ODY2MzYiIGlkPSJGaWxsLTEiPjwvcGF0aD4KICAgICAgICAgICAgICAgICAgICA8cGF0aCBkPSJNMS40MjEyMTY5NSwxOS44NTE0NjQ4IEMxLjIyMjkzNzI0LDIwLjA0OTUxMTcgMC44OTg3NTc1NjYsMjAuMDQ5NTExNyAwLjcwMDIyMzAwMSwxOS44NTE0NjQ4IEwwLjE0ODcwOTc4LDE5LjI5OTg5MDEgQy0wLjA0OTU2OTkyNjcsMTkuMTAxODQzMSAtMC4wNDk1Njk5MjY3LDE4Ljc3NzExNzYgMC4xNDg3MDk3OCwxOC41NzkwNzA2IEwyLjU3OTAzNzkxLDE2LjE0ODcyNjQgQzIuNzc3MzE3NjIsMTUuOTUwNDI0NSAzLjEwMTQ5NzI5LDE1Ljk1MDQyNDUgMy4yOTk3NzcsMTYuMTQ4NzI2NCBMMy44NTEyOTAyMiwxNi43MDAzMDExIEM0LjA0OTU2OTkzLDE2Ljg5ODM0OCA0LjA0OTU2OTkzLDE3LjIyMjgxODcgMy44NTEyOTAyMiwxNy40MjExMjA1IEwxLjQyMTIxNjk1LDE5Ljg1MTQ2NDggWiIgaWQ9IkZpbGwtNCI+PC9wYXRoPgogICAgICAgICAgICAgICAgPC9nPgogICAgICAgICAgICA8L2c+CiAgICAgICAgPC9nPgogICAgPC9nPgo8L3N2Zz4=' );

        //call register settings function
        add_action( 'admin_init', function() {
            register_setting( 'schoolwiki', 'schoolwiki_domain' );
            register_setting( 'schoolwiki', 'schoolwiki_school' );
            register_setting( 'schoolwiki', 'schoolwiki_color' );
            register_setting( 'schoolwiki', 'schoolwiki_title' );
            register_setting( 'schoolwiki', 'schoolwiki_position' );
            add_filter('plugin_action_links_'.plugin_basename(__FILE__), function( $links ) {
                $links[] = '<a href="' .admin_url( 'admin.php?page=schoolwiki' ) .'">' . __('Settings') . '</a>';
                 return $links;
            });
        } );
    });
  }

	/**
	 * Check if the plugin meets requirements and
	 * disable it if they are not present.
	 *
	 * @since  1.0.0
	 *
	 * @return boolean True if requirements met, false if not.
	 */
	public function check_requirements() {

		// Bail early if plugin meets requirements.
		if ( $this->meets_requirements() ) {
			return true;
		}


		// Deactivate our plugin.
		add_action( 'admin_init', array( $this, 'deactivate_me' ) );

		// Didn't meet the requirements.
		return false;
	}

	/**
	 * Deactivates this plugin, hook this function on admin_init.
	 *
	 * @since  1.0.0
	 */
	public function deactivate_me() {

		// We do a check for deactivate_plugins before calling it, to protect
		// any developers from accidentally calling it too early and breaking things.
		if ( function_exists( 'deactivate_plugins' ) ) {
			deactivate_plugins( $this->basename );
		}
	}

	/**
	 * Adds a notice to the dashboard if the plugin requirements are not met.
	 *
	 * @since  1.0.0
	 */
	public function requirements_not_met_notice() {
		?>
		<div id="message" class="error">
			<p><b>Schoolwiki widget configuratie ontbreekt</b><br />Ga naar de <a href="admin.php?page=schoolwiki">instellingen pagina</a> om de installatie van het widget af te ronden.</p>
		</div>
		<?php
	}

	/**
	 * Magic getter for our object.
	 *
	 * @since  1.0.0
	 *
	 * @param  string $field Field to get.
	 * @throws Exception     Throws an exception if the field is invalid.
	 * @return mixed         Value of the field.
	 */
	public function __get( $field ) {
		switch ( $field ) {
			case 'version':
				return self::VERSION;
			case 'basename':
			case 'url':
			case 'path':
			case 'options':
				return $this->$field;
			default:
				throw new Exception( 'Invalid ' . __CLASS__ . ' property: ' . $field );
		}
	}

	/**
	 * Include a file from the includes directory.
	 *
	 * @since  1.0.0
	 *
	 * @param  string $filename Name of the file to be included.
	 * @return boolean          Result of include call.
	 */
	public static function include_file( $filename ) {
		$file = self::dir( $filename . '.php' );
		if ( file_exists( $file ) ) {
			return include_once( $file );
		}
		return false;
	}

	/**
	 * This plugin's directory.
	 *
	 * @since  1.0.0
	 *
	 * @param  string $path (optional) appended path.
	 * @return string       Directory and path.
	 */
	public static function dir( $path = '' ) {
		static $dir;
		$dir = $dir ? $dir : trailingslashit( dirname( __FILE__ ) );
		return $dir . $path;
	}

	/**
	 * This plugin's url.
	 *
	 * @since  1.0.0
	 *
	 * @param  string $path (optional) appended path.
	 * @return string       URL and path.
	 */
	public static function url( $path = '' ) {
		static $url;
		$url = $url ? $url : trailingslashit( plugin_dir_url( __FILE__ ) );
		return $url . $path;
	}
}

/**
 * Grab the Schoolwiki object and return it.
 * Wrapper for Schoolwiki::get_instance().
 *
 * @since  1.0.0
 * @return Schoolwiki  Singleton instance of plugin class.
 */
function schoolwiki() {
	return Schoolwiki::get_instance();
}

// Kick it off.
add_action( 'plugins_loaded', array( schoolwiki(), 'hooks' ) );

// Activation and deactivation.
register_activation_hook( __FILE__, array( schoolwiki(), '_activate' ) );
register_deactivation_hook( __FILE__, array( schoolwiki(), '_deactivate' ) );
